<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Services\CRMGLService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Exception;

class ContactController extends Controller
{
    protected CRMGLService $crmService;

    public function __construct()
    {
        $this->crmService = new CRMGLService();
        
        // Configurar el servicio con las variables del .env
        $this->crmService->setToken(env('CRM_LCMEDIA_GLAPI_APIKEY'));
        $this->crmService->setLocationId(env('CRM_LCMEDIA_GLAPI_Location'));
    }

    /**
     * Obtener información de un contacto de GoHighLevel por su ID
     *
     * @param Request $request
     * @param string $contactId
     * @return JsonResponse
     */
    public function getContact(Request $request, string $contactId): JsonResponse
    {
        try {
            Log::info('Fetching contact from GHL', [
                'contact_id' => $contactId,
                'ip' => $request->ip(),
                'origin' => $request->header('Origin'),
            ]);

            $contact = $this->crmService->getContactById($contactId);

            return response()->json([
                'success' => true,
                'data' => $contact,
            ], 200)->withHeaders([
                'Access-Control-Allow-Origin' => '*',
                'Access-Control-Allow-Methods' => 'GET, POST, PUT, DELETE, OPTIONS',
                'Access-Control-Allow-Headers' => 'Content-Type, Authorization, X-Requested-With',
                'Access-Control-Max-Age' => '3600',
            ]);

        } catch (Exception $e) {
            Log::error('Error fetching contact from GHL', [
                'contact_id' => $contactId,
                'error' => $e->getMessage(),
                'ip' => $request->ip(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error al obtener el contacto: ' . $e->getMessage(),
            ], 500)->withHeaders([
                'Access-Control-Allow-Origin' => '*',
                'Access-Control-Allow-Methods' => 'GET, POST, PUT, DELETE, OPTIONS',
                'Access-Control-Allow-Headers' => 'Content-Type, Authorization, X-Requested-With',
            ]);
        }
    }

    /**
     * Manejar solicitudes OPTIONS (preflight)
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function options(Request $request): JsonResponse
    {
        return response()->json([], 200)->withHeaders([
            'Access-Control-Allow-Origin' => '*',
            'Access-Control-Allow-Methods' => 'GET, POST, PUT, DELETE, OPTIONS',
            'Access-Control-Allow-Headers' => 'Content-Type, Authorization, X-Requested-With',
            'Access-Control-Max-Age' => '3600',
        ]);
    }
}
